#############################
# 
#   rnaseq_utils.R
#
# Utility R functions for RNA-seq processing pipeline
# These functions are mainly designed to assist in EdgeR analysis from RNA-seq count data
# (e.g. starting from gene-level counts generated by HTSeq-count)
#
# Author: Logan J Everett, PhD
# Email: logan_everett@ncsu.edu
# Created on: 5/27/15
# Updated: 5/27/15
#

# For EdgeR pipeline functions, goodness of fit function
library(edgeR)

# For plotRLE and plotPCA functions
library(EDASeq)



# TO DO:
# Helper functions to build count table from multiple HTSeq-count output tables
# And to build sample info table from multiple batch tables

# Function to down-sample feature counts
# Makes a vector of each feature duplicated X times, with X = read count, which then allows for sampling without replacement
# When feature.counts contains ALL features from HTSeq-count (including ambiguous, unaligned),
# Then this should be equivalent to randomly downsampling the aligned BAM file
# To get reproducible results, make sure to seed the RNG with set.seed()
down.sample.counts <- function(feature.counts, sample.size) {
  # feature.counts should be a named vector (otherwise default names will be assigned)
  if(is.null(names(feature.counts))) {
    names(feature.counts) <- paste0("Feature", 1:length(feature.counts))
  }
  # sample.size should be < sum(feature.counts)
  if(sum(feature.counts) <= sample.size) {
    warning("sample.size should be less than sum of feature.counts")
  }
  # Set up feature vector to sample from
  feature.reps <- rep(names(feature.counts), times=feature.counts)
  # Sample features according to depth
  feature.samples <- sample(x=feature.reps, size=as.integer(sample.size), replace=F)
  feature.sample.counts <- c(table(feature.samples)[names(feature.counts)],recursive=T)
  feature.sample.counts[is.na(feature.sample.counts)] <- 0
  names(feature.sample.counts) <- names(feature.counts)
  return(feature.sample.counts)
}

# TO DO: A function to apply down.sample.counts to all sample columns of a table


# --- QC Plotting Functions --- #

# These are primarily used to identify major normalization issues and problematic samples

# For each of these functions, count.data should be a gene by sample count matrix or DGEList object, with row names matching columns in same order as sample.info
# TO DO: NO! All other information is taken from global variables (sample.info, sample.colors, output.path)

# Draw RLE plot:
# This draws a box plot of Relative Log Expression values for each sample
# Ideal output is that each sample distribution is centered on 0 with similar distribution for each sample
# If count.data is a DGEList, expression values will be normalized to library size (including normalization factors if computed already)
QC.plot.RLE <- function(
  count.data, 
  sample.colors=rep("white", times=ncol(count.data)), # Optional color coding, e.g. for sample batches or types
  plot.file="",   # Set to a file name (including path) to output to a pdf file, otherwise this draws within the R session
  lwd=0.5, # Looks better for large numbers of samples, can set to 1 for small batches
  xaxt="n", # Suppress X axis by default (generally looks bad for more than a few samples)
  ...   # Additional parameters passed directly to plotRLE function (e.g. main)
) {
  if(class(count.data) == "DGEList") {
    count.data <- cpm(count.data, normalized.lib.sizes=T)
  }
  if(plot.file != "") {
    pdf(plot.file)
  }
  plotRLE(count.data, outline=F, col=sample.colors, xaxt=xaxt, ylab="RLE Distribution", lwd=lwd, ...)
  if(plot.file != "") {
    dev.off()
  }
}

# Draw PCA plot:
# This computes the first two principle componenets for total expression, and plots samples on PC1 x PC2 plot
# Ideal output is that samples are evenly distributed in this space, without any visible batch or other unexpected effects
# If count.data is a DGEList, expression values will be normalized to library size (including normalization factors if computed already)
# TO DO: Write a custom function that computes the PCs here, reports the variance captured, and draws the plot with symbols instead of text
QC.plot.PCA <- function(
  count.data,
  sample.colors=rep("white", times=ncol(count.data)), # Optional color coding, e.g. for sample batches or types
  sample.names=colnames(count.data), # Optional shorter names (short names work best for the PCA plot)
  plot.file="",   # Set to a file name (including path) to output to a pdf file, otherwise this draws within the R session
  cex=0.5,  # Shrinking the sample name text is generally more legible
  ...
) {
  if(class(count.data) == "DGEList") {
    count.data <- cpm(count.data, normalized.lib.sizes=T)
  }
  colnames(count.data) <- sample.names
  if(plot.file != "") {
    pdf(plot.file)
  }
  plotPCA(count.data, col=sample.colors, cex=cex, ...)
  if(plot.file != "") {
    dev.off()
  }
}

# Draw RPKM boxplot (Automatically normalizes to depth and length)
# If count.data is a matrix, normalizes to column totals and gene.lengths (must be specified)
# If count.data is a DGEList, then it calls rpkm with default params (DGEList object needs to have been constructed with a LENGTH column in genes)
QC.plot.RPKM <- function(
  count.data, 
  gene.lengths=NULL, # Required if count.data is a count matrix rather than DGEList
  sample.colors=rep("white", times=ncol(count.data)), # Optional color coding, e.g. for sample batches or types
  xaxt="n", # Suppress x axis by default, can't fit lots of sample names down there
  lwd=0.5,  # Looks better for lots of samples, set to 1 for smaller number of samples
  plot.file="",
  ... # Additional params passed to boxplot function
) {
  if(class(count.data) == "matrix") {
    stopifnot(!is.null(gene.lengths))
    stopifnot(length(gene.lengths) == nrow(count.data))
    count.sums <- as.numeric(apply(count.data, 2, sum))
    for(j in 1:ncol(count.data)) {
      count.data[,j] <- count.data[,j] * (10**9) / (count.sums[j] * gene.lengths)
    }
  } else if(class(count.data) == "DGEList") {
    count.data <- rpkm(count.data)
  } else {
    stop("Invalid count.data class: ", class(count.data))
  }
  if(plot.file != "") {
    pdf(plot.file)
  }
  boxplot(count.data, outline=F, col=sample.colors, xaxt=xaxt, ylab="RPKM Distribution", lwd=lwd, ...)
  if(plot.file != "") {
    dev.off()
  }
}

# Generate all plots above, output to files named (filestub)_RLE.pdf, etc.
# This version optionally accepts a data frame for count.data, and does the conversion to matrix for you (Count matrix or DGEList also still work)
# This also takes all sample info (short names for PCA, color coding) from a single table

QC.plots <- function(
  count.data,
  sample.info=NULL, # Provide a table with rows matching columns of count.data
  sample.name.col=NULL, # Set to a column name containing shorter sample names appropriate for PCA plot
  sample.color.col=NULL,  # Set to a column used for coloration
  sample.color.map=NULL,  # Named vector mapping values in sample.pca.color.col to desired colors (e.g. for batch or sex coloration)
  gene.lengths=NULL,  # Passed to RPKM plot only
  filestub="QC_default",  # Used for all file names
  output.path="", # Set an optional path (can also just be included in filestub)
  ... # Additional params passed to ALL plotting functions
) {
  if(class(count.data) == "data.frame") {
    count.data <- as.matrix(count.data)
  }
  # Pull information from sample.info if specified
  if(!is.null(sample.info)) {
    if(!is.null(sample.name.col)) {
      sample.names <- sample.info[,sample.name.col]
    } else {
      sample.names <- colnames(count.data)
    }
    if(!is.null(sample.color.col)) {
      sample.colors <- sample.info[,sample.color.col]
      if(!is.null(sample.color.map)) {
        sample.colors <- sample.color.map[sample.colors]
      }
    } else {
      sample.colors <- rep("white", times=ncol(count.data))
    }
  } else {
    sample.names <- colnames(count.data)
    sample.colors <- rep("white", times=ncol(count.data))
  }
  
  # Put output.path at the front of filestub
  if(output.path != "") {
    filestub <- paste(output.path, filestub, sep="/")
  }
  QC.plot.RLE(count.data, sample.colors=sample.colors, plot.file=paste(filestub, "_RLE.pdf", sep=""), ...)
  QC.plot.PCA(count.data, sample.colors=sample.colors, sample.names=sample.names, plot.file=paste(filestub, "_PCA.pdf", sep=""), ...)
  QC.plot.RPKM(count.data, sample.colors=sample.colors, gene.lengths=gene.lengths, plot.file=paste(filestub, "_RPKM.pdf", sep=""), ...)
}


# TO DO: EdgeR main pipeline functions
# Current design is to split into two steps:
# 1) General set up function: set up the DGEList object, do all dispersion estimation, fit the GLM, output some QC plots along the way
# 2) Pairwise test function: Given the DGEFLM object, perform LRT test for a specific contrast or coefficient, output some plots and result table along the way
